#include <windows.h>
#include <ddraw.h>
#include <stdio.h>
#include "video.h"
#include <fcntl.h>
#include <io.h>
#include <sys/stat.h>


static LPDIRECTDRAW7         g_pDD;  		// DirectDraw object
static LPDIRECTDRAWSURFACE7  g_pDDSPrimary;	 // DirectDraw primary surface
static LPDIRECTDRAWSURFACE7    g_pDDSBack;		// DirectDraw back surface
static RECT                    g_rcWindow;      // Saves the window size & pos.
static RECT                    g_rcViewport;    // Pos. & size to blt from
static RECT                    g_rcScreen;      // Screen pos. for blt

static bool g_bWindowed;

static DDSURFACEDESC2      ddsd;
static DDSCAPS2            ddscaps;

static int bpp;
static int bytespp;
static unsigned char* pixels;
static int pitch;
static int start_x, max_x;
static int start_y, max_y;
static HWND	myWindow;	// surface window
static int srcwidth;
static int srcheight;

static bool first_time_clear;

static DWORD r_arr[256];
static DWORD g_arr[256];
static DWORD b_arr[256];

static DWORD r_bitmask;
static DWORD g_bitmask;
static DWORD b_bitmask;

static int InitDdrawSurfaces(bool windowed);

extern bool always_on_top;
extern bool aspect_ratio;

static int r_shift;
static int g_shift;
static int b_shift;

static int r_cnt;
static int g_cnt;
static int b_cnt;

extern bool use_double_buffering;


static void findShiftValue(DWORD mask, int &shft, int& cnt)
{
	int t;
	for (t = 0; t < 32; t++)
   	if (mask & (1<<t))
      	break;
   shft = t;

   cnt = 0;
   while (mask & (1<<t))
   {
   	t++;
      cnt++;
   }
}

static void init_rgb_array(DWORD dwRBitMask, DWORD dwGBitMask, DWORD dwBBitMask)
{
	r_bitmask = dwRBitMask;
	g_bitmask = dwGBitMask;
	b_bitmask = dwBBitMask;

   findShiftValue(r_bitmask, r_shift, r_cnt);
   findShiftValue(g_bitmask, g_shift, g_cnt);
   findShiftValue(b_bitmask, b_shift, b_cnt);

	for (__int64 i = 0; i < 256; i++)
  	{
  		r_arr[i] = (DWORD)(((i * __int64(dwRBitMask)) / 255) & dwRBitMask);
   	g_arr[i] = (DWORD)(((i * __int64(dwGBitMask)) / 255) & dwGBitMask);
  		b_arr[i] = (DWORD)(((i * __int64(dwBBitMask)) / 255) & dwBBitMask);
   }
}

#define MAKERGB(r,g,b) (r_arr[r]|g_arr[g]|b_arr[b])


static void ReleaseAllObjects(void)
{
    if (g_pDD != NULL)
    {
        g_pDD->SetCooperativeLevel(myWindow, DDSCL_NORMAL);
        if (g_pDDSBack != NULL)
        {
            g_pDDSBack->Release();
            g_pDDSBack = NULL;
        }
        if (g_pDDSPrimary != NULL)
        {
            g_pDDSPrimary->Release();
            g_pDDSPrimary = NULL;
        }
    }
}


int initDdraw(HWND hwnd, char* errorbuf)
{
	HRESULT hRet;
   // LPDIRECTDRAW    pDD;            // DD1 interface, used to get DD4 interface

   myWindow = hwnd;

	hRet = DirectDrawCreateEx(NULL, (VOID**)&g_pDD, IID_IDirectDraw7, NULL);
	if (hRet!= DD_OK)
   {
   	strcpy(errorbuf, "No puedo iniciar DirectDraw");
   	return(-1);
   }

    // Fetch DirectDraw4 interface
  // hRet = pDD->QueryInterface(IID_IDirectDraw, (LPVOID *)&g_pDD);
  //if (FAILED(hRet))
  // {
  // 	strcpy(errorbuf, "QueryInterface FALLO");
  // 	return(-1);
  // }

   // Initialize all the surfaces we need
   if (InitDdrawSurfaces(true) < 0)
   {
   	strcpy(errorbuf, "InitSurfaces FALLO");
   	return(-1);
   }

   return(0);
}


/*
 Free ddraw resources
*/
void uninitDdraw(void)
{
	ReleaseAllObjects();

	if (g_pDD != NULL)
	{
		g_pDD->Release();
		g_pDD = NULL;
	}
}


//-----------------------------------------------------------------------------
// Name: InitSurfaces()
// Desc: Create all the needed DDraw surfaces and set the coop level
//-----------------------------------------------------------------------------
static int InitDdrawSurfaces(bool windowed)
{
	HRESULT		         hRet;
	LPDIRECTDRAWCLIPPER pClipper;

	g_bWindowed = windowed;

	if (g_bWindowed)
	{	
          
		GetWindowRect(myWindow, &g_rcWindow);

		// Get normal windowed mode
		hRet = g_pDD->SetCooperativeLevel(myWindow, DDSCL_NORMAL);
		if (hRet != DD_OK)
      		return(-1);

		// Create the primary surface
		ZeroMemory(&ddsd,sizeof(ddsd));
		
		ddsd.dwSize = sizeof(ddsd);
		ddsd.dwFlags = DDSD_CAPS;
		ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE;
		hRet = g_pDD->CreateSurface(&ddsd, &g_pDDSPrimary, NULL);
		if (hRet != DD_OK)
      		return(-1);

		// Create a clipper object since this is for a Windowed render
		hRet = g_pDD->CreateClipper(0, &pClipper, NULL);
		if (hRet != DD_OK)
      		return(-1);

		// Associate the clipper with the window
		pClipper->SetHWnd(0, myWindow);
		g_pDDSPrimary->SetClipper(pClipper);
		pClipper->Release();
		pClipper = NULL;

		// Get the backbuffer. For fullscreen mode, the backbuffer was created
		// along with the primary, but windowed mode still needs to create one.
		ddsd.dwFlags        = DDSD_WIDTH | DDSD_HEIGHT | DDSD_CAPS;
		ddsd.dwWidth        = 800;
		ddsd.dwHeight       = 600;
		ddsd.ddsCaps.dwCaps = DDSCAPS_OFFSCREENPLAIN;
		hRet = g_pDD->CreateSurface(&ddsd, &g_pDDSBack, NULL);
		if (hRet != DD_OK)
      		return(-1);
	}
	else
    {
		 
            
           

		// Get exclusive mode
		hRet = g_pDD->SetCooperativeLevel(myWindow, DDSCL_EXCLUSIVE|DDSCL_FULLSCREEN);
		if (hRet != DD_OK)
			return(-1); 
		
		// Set the video mode to ...
		hRet = g_pDD->SetDisplayMode(800,600,16,0,0);   // , 0, 0);
		if (hRet != DD_OK)
      		return(-1);
		
		SetRect(&g_rcScreen, 0, 0, 800, 600);

		// Create the primary surface with 1 back buffer
		ZeroMemory(&ddsd,sizeof(ddsd));
		ddsd.dwSize = sizeof(ddsd);
		ddsd.dwFlags = DDSD_CAPS | DDSD_BACKBUFFERCOUNT;
		ddsd.ddsCaps.dwCaps = DDSCAPS_PRIMARYSURFACE |
                 			  DDSCAPS_FLIP |
                			  DDSCAPS_COMPLEX;

		ddsd.dwBackBufferCount = 1;
		
		
		hRet = g_pDD->CreateSurface( &ddsd, &g_pDDSPrimary, NULL);
		if (hRet != DD_OK)
      		return(-1);
		
		ddscaps.dwCaps = DDSCAPS_BACKBUFFER;
		
		
		hRet = g_pDDSPrimary->GetAttachedSurface(&ddscaps, &g_pDDSBack);

		
		if (hRet != DD_OK)
      		return(-1);

	}

	DDPIXELFORMAT pf;
	memset(&pf, 0, sizeof(pf));
	pf.dwSize = sizeof(pf);
	g_pDDSPrimary->GetPixelFormat(&pf);

	bpp = pf.dwRGBBitCount;
	bytespp = (bpp + 7) / 8;

	init_rgb_array(pf.dwRBitMask, pf.dwGBitMask, pf.dwBBitMask);

	return(0);
}


int DdrawChangeCoopLevel(bool windowed)
{
    ReleaseAllObjects();

    if (windowed)
    {
        SetWindowPos(myWindow,
          (always_on_top)?HWND_TOPMOST:HWND_NOTOPMOST,
          g_rcWindow.left, g_rcWindow.top,
          (g_rcWindow.right - g_rcWindow.left),
          (g_rcWindow.bottom - g_rcWindow.top), SWP_SHOWWINDOW );
    }

    int ret = InitDdrawSurfaces(windowed);

    DdrawWindowChanged();

  	ShowCursor(windowed);
	
    first_time_clear = true;

    return(ret);
}


void DdrawsetSrcSize(int swidth, int sheight)
{
	if (swidth > 0 && sheight > 0)
	{
	 	srcwidth = swidth;
   		srcheight = sheight;
		first_time_clear = true;
	}

	SetRect(&g_rcViewport, 0, 0, srcwidth, srcheight);

	if (g_bWindowed)
	{
   		SetWindowPos(myWindow,
			(always_on_top)?HWND_TOPMOST:HWND_NOTOPMOST,
			g_rcWindow.left, g_rcWindow.top,
   			srcwidth+GetSystemMetrics(SM_CXSIZEFRAME)*2,
			srcheight+GetSystemMetrics(SM_CYSIZEFRAME)*2+GetSystemMetrics(SM_CYMENU)*2,
			SWP_SHOWWINDOW);
	}
	DdrawWindowChanged();   
}


void DdrawCheckAspectRatio(void)
{
	if (g_bWindowed)
	{
		GetWindowRect(myWindow, &g_rcWindow);
		int h = g_rcWindow.bottom - g_rcWindow.top
			- GetSystemMetrics(SM_CYSIZEFRAME)*2-GetSystemMetrics(SM_CYMENU)*2;
		int w = h * srcwidth / srcheight;
	
   		SetWindowPos(myWindow,
			(always_on_top)?HWND_TOPMOST:HWND_NOTOPMOST,
			g_rcWindow.left, g_rcWindow.top,
   			w+GetSystemMetrics(SM_CXSIZEFRAME)*2,
			h+GetSystemMetrics(SM_CYSIZEFRAME)*2+GetSystemMetrics(SM_CYMENU)*2,
			SWP_SHOWWINDOW);

   		DdrawWindowChanged();
	}
}


void DdrawTranslateToSrcRect(int& x, int &y)
{
	int h, w;

	if (g_bWindowed)
	{
		RECT rect;
		GetClientRect(myWindow, &rect);
		h = rect.bottom - rect.top;
		w = rect.right - rect.left;
		if (w == 0) w = 1;
		if (h == 0) h = 1;
	}
	else
	{
		w = 800;
		h = 600;
	}

	x = (x * srcwidth) / w;
	y = (y * srcheight) / h;
}


void DdrawWindowChanged(void)
{
	if (g_bWindowed)
	{
		GetWindowRect(myWindow, &g_rcWindow);
		GetClientRect(myWindow, &g_rcScreen);
		ClientToScreen(myWindow, (POINT*)&g_rcScreen.left);
		ClientToScreen(myWindow, (POINT*)&g_rcScreen.right);
	}

  	if (use_double_buffering)
	{
   		start_x = 0;
		start_y = 0;
   		max_x = 800;
		max_y = 600;
	}
	else
	{
   		if (g_bWindowed)
		{
	   		start_x = g_rcScreen.left;
			start_y = g_rcScreen.top;
		}
		else
		{
      		start_x = (800 - srcwidth) / 2;
			start_y = (600 - srcheight) / 2;
		}

		max_x = GetSystemMetrics(SM_CXFULLSCREEN);
		max_y = GetSystemMetrics(SM_CYFULLSCREEN);
	}
}


void DdrawCopyBackToFront(void)
{
	HRESULT hRet;

	if (!use_double_buffering)
   		return;

	while (1)
	{
		hRet = g_pDDSPrimary->Blt(&g_rcScreen, g_pDDSBack,
			&g_rcViewport, DDBLT_WAIT, NULL);
		if (hRet == DD_OK )
			break;
		if (hRet == DDERR_SURFACELOST )
		{
			hRet = g_pDDSPrimary->Restore();
			if (hRet != DD_OK )
         		break;
		}
		if (hRet != DDERR_WASSTILLDRAWING )
			break;
	}
}


/*
 Lock display, get pointer to video memory in "pixels"
*/
void lockDisplay(void)
{
	int ddrval;

	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_PITCH;

	while (true)
	{
   		if (use_double_buffering)
			ddrval = g_pDDSBack->Lock(NULL, &ddsd, DDLOCK_WAIT, NULL );
		else
      		ddrval = g_pDDSPrimary->Lock(NULL, &ddsd, DDLOCK_WAIT, NULL );

		if (ddrval == DD_OK)
			break;
		if (ddrval == DDERR_SURFACELOST)
			g_pDDSBack->Restore();
		else
      		exit(-1);
	}

	pitch = ddsd.lPitch;
	pixels = (unsigned char*)ddsd.lpSurface;
}

/*
 Unlock the display (stop drawing)
*/
void unlockDisplay(void)
{
	if (pixels != NULL)
	{
		pixels = NULL;

   		if (use_double_buffering)
 			g_pDDSBack->Unlock(NULL);
		else
 			g_pDDSPrimary->Unlock(NULL);
	}
}


void clearDisplay(void)
{
	if (!use_double_buffering && !first_time_clear)
   	return;
   first_time_clear = false;

	// Use the blter to do a color fill to clear the back buffer
   DDBLTFX     ddbltfx;
   ZeroMemory(&ddbltfx, sizeof(ddbltfx));
   ddbltfx.dwSize = sizeof(ddbltfx);
   ddbltfx.dwFillColor = 0;
  	if (use_double_buffering)
	   g_pDDSBack->Blt(NULL, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &ddbltfx);
	else
   	g_pDDSPrimary->Blt(NULL, NULL, NULL, DDBLT_COLORFILL | DDBLT_WAIT, &ddbltfx);
}


void putPixel(int x, int y, unsigned char red, unsigned char green, unsigned char blue)
{
   x += start_x;
   y += start_y;
   if (x < 0 || x > max_x || y < 0 || y > max_y)
   	return;

   lockDisplay();

   unsigned char* surface = pixels + pitch * y + x * bytespp;

   if (bpp == 16)
	   *(unsigned short *)surface = (unsigned short)MAKERGB(red, green, blue);
   else if (bpp == 32)
	   *(unsigned long *)surface = (unsigned long)MAKERGB(red, green, blue);

   unlockDisplay();
}


void drawText(int x, int y, const char* s, COLORREF color)
{
	HDC hdc;
	HRESULT ret;

   if (use_double_buffering)
   	ret = g_pDDSBack->GetDC(&hdc);
   else
   	ret = g_pDDSPrimary->GetDC(&hdc);

   x += start_x;
   y += start_y;

	if (ret == DD_OK)
   {
   	SetBkMode(hdc, TRANSPARENT);
      SetTextColor(hdc, color);

      RECT rect;
      rect.left = x;
      rect.top = y;
      rect.right = 800-x;
      rect.bottom = 600 - y;
		DrawText(hdc, s, strlen(s), &rect, 0);

	   if (use_double_buffering)
   		g_pDDSBack->ReleaseDC(hdc);
	   else
   		g_pDDSPrimary->ReleaseDC(hdc);
   }
}


unsigned char* getPixelPtr(int& thepitch, int& thebpp)
{
	thepitch = pitch;
	thebpp = bpp;
	return(pixels);
}


void drawPixels(int x, int y, RGBTRIPLE* rgbpixels, int nbr_pixels)
{
	x += start_x;
	y += start_y;

//	if (y < 0 || y > max_y)
//   		return;
//	if (x + nbr_pixels > max_x)
//  		nbr_pixels = max_x - x;
//	if (x < 0)
//	{
//   		int r = 0-x;
//   		rgbpixels += r;
//		nbr_pixels -= r;
//		x = 0;
//	}

	unsigned char* surface = pixels + y * pitch + x * bytespp;

  	if (bpp == 16)
	{
		register unsigned short *s0 = (unsigned short*)surface;
   		while (--nbr_pixels > 0)
   		{
			*s0++ = (unsigned short)MAKERGB(rgbpixels->rgbtRed, rgbpixels->rgbtGreen, rgbpixels->rgbtBlue);
			rgbpixels++;
		}
	}
  	else if (bpp == 32)
	{
		register unsigned long *s0 = (unsigned long*)surface;
   		while (--nbr_pixels > 0)
   		{
			*s0++ = MAKERGB(rgbpixels->rgbtRed, rgbpixels->rgbtGreen, rgbpixels->rgbtBlue);
     		rgbpixels++;
		}
	}
}


typedef struct {
    BYTE    rgbtGreen;
    BYTE    rgbtRed;
    BYTE    rgbtBlue;
} MYRGBTRIPLE;


int writeBMP(const char* filename)
{
   unlink(filename);
   int fd = open(filename, O_CREAT|O_WRONLY|O_BINARY,S_IREAD|S_IWRITE);
   if (fd == -1)
   	return(-1);

   BITMAPFILEHEADER head;
   memset(&head, 0, sizeof(head));
   BITMAPINFOHEADER head2;
   memset(&head2, 0, sizeof(head2));
	MYRGBTRIPLE pixel[800];		// > srcwidth

   memcpy(&head.bfType, "BM", 2);
   head.bfSize = sizeof(head)+sizeof(head2)+sizeof(MYRGBTRIPLE)*400*300;
   head.bfReserved1 = 0;
   head.bfReserved2 = 0;
   head.bfOffBits = sizeof(BITMAPINFO);
   head2.biSize  = sizeof(head2);
   head2.biWidth = srcwidth;
   head2.biHeight = srcheight;
   head2.biPlanes = 1;
   head2.biBitCount = 24;
   head2.biCompression = BI_RGB;
   head2.biSizeImage = 0;
   head2.biXPelsPerMeter = 300;
   head2.biYPelsPerMeter = 300;
   head2.biClrUsed = 0;
   head2.biClrImportant = 0;

   write(fd, &head, sizeof(head));
   write(fd, &head2, sizeof(head2));

	lockDisplay();
   // We moeten van onder naar boven schrijven (BMP formaat ...)
   unsigned char *surface = pixels+srcheight*pitch;
   for (int y = 0; y < srcheight; y++)
   {
      surface -= pitch;

//	   if (bpp == 16)
//      {
      	int rs =  8 - r_cnt;
      	int gs =  8 - g_cnt;
      	int bs =  8 - b_cnt;

	   	unsigned short *ptr = (unsigned short*)surface;
   	   for (int x = 0; x < srcwidth; x++)
	   	{
   			unsigned short rgb = *ptr++;
		   	pixel[x].rgbtRed = (unsigned char)((rgb&r_bitmask) >> r_shift) << rs;
			pixel[x].rgbtGreen = (unsigned char)((rgb&g_bitmask) >> g_shift) << gs;
		   	pixel[x].rgbtBlue = (unsigned char)((rgb&b_bitmask) >> b_shift) << bs;
	      }
//      }
//      else if (bpp == 32)
//	  {
//	   	unsigned long *ptr = (unsigned long*)surface;
//   	   for (int x = 0; x < srcwidth; x++)
//	   	{
//   			unsigned long rgb = *ptr++;
//		   	pixel[x].rgbtRed = (unsigned char)((rgb&r_bitmask) >> r_shift);
//			pixel[x].rgbtGreen = (unsigned char)((rgb&g_bitmask) >> g_shift);
//		   	pixel[x].rgbtBlue = (unsigned char)((rgb&b_bitmask) >> b_shift);
//	      }
//      }

   	write(fd, pixel, sizeof(MYRGBTRIPLE)*srcwidth);
   }

   unlockDisplay();

   close(fd);

   return(0);
}


int writeBMP(const char* filename, unsigned char* raw_data,
  int width, int height)
{
   unlink(filename);
   int fd = open(filename, O_CREAT|O_WRONLY|O_BINARY,S_IREAD|S_IWRITE);
   if (fd == -1)
   	return(-1);

   BITMAPFILEHEADER head;
   memset(&head, 0, sizeof(head));
   BITMAPINFOHEADER head2;
   memset(&head2, 0, sizeof(head2));

   MYRGBTRIPLE* pixel = (MYRGBTRIPLE*)malloc(sizeof(MYRGBTRIPLE)*width);

   memcpy(&head.bfType, "BM", 2);
   head.bfSize = sizeof(head)+sizeof(head2)+sizeof(MYRGBTRIPLE)*400*300;
   head.bfReserved1 = 0;
	head.bfReserved2 = 0;
	head.bfOffBits = sizeof(BITMAPINFO);
   head2.biSize  = sizeof(head2);
   head2.biWidth = width;
   head2.biHeight = height;
   head2.biPlanes = 1;
   head2.biBitCount = 24;
   head2.biCompression = BI_RGB;
   head2.biSizeImage = 0;
   head2.biXPelsPerMeter = 0;
   head2.biYPelsPerMeter = 0;
   head2.biClrUsed = 0;
   head2.biClrImportant = 0;

   write(fd, &head, sizeof(head));
   write(fd, &head2, sizeof(head2));

   // We moeten van onder naar boven schrijven (BMP formaat ...)
   raw_data += height*width;
   for (int y = 0; y < height; y++)
   {
      raw_data -= width;
  	   for (int x = 0; x < width; x++)
   	{
	   	pixel[x].rgbtRed = pixel[x].rgbtGreen = pixel[x].rgbtBlue = raw_data[x];
      }

   	write(fd, pixel, sizeof(MYRGBTRIPLE)*width);
   }

   close(fd);

   free(pixel);

   return(0);
}
